/*
 * Created on 10.01.2005
 */
package q1471341.mp1074.test;

import javax.swing.JTextArea;

import q1471341.mp1074.integration.Integrand;
import q1471341.mp1074.integration.IntegrationResult;
import q1471341.mp1074.integration.MonteCarloIntegrator;
import q1471341.mp1074.integration.lattice.FAlpha;
import q1471341.mp1074.integration.lattice.GoodLatticePointsRule;
import q1471341.mp1074.integration.lattice.EmbeddedCopyRule;
import q1471341.mp1074.integration.lattice.Optimizer;
import q1471341.mp1074.integration.lattice.Polynom1Periodizer;
import q1471341.mp1074.integration.lattice.Polynom2Periodizer;
import q1471341.mp1074.integration.lattice.Trigonometric1Periodizer;
import q1471341.util.PrintfFormat;

/**
 * This package provides several tests for the integration framework.
 * 
 * @author Ulrich Telle
 */
public class TestFactory {

	/**
	 * Precalculated generating vectors for the method of good lattice points
	 */
	static int[][][] zz_glp = {
			{
			  { 1 },
			  { 1 },
			  { 1, 1939 },
			  { 1, 1611, 3767 },
			  { 1, 1659, 631, 1202 },
			  { 1, 2384, 48, 4366, 2304 },
			  { 1, 1480, 4089, 3093, 4898, 4696 },
			  { 1, 2009, 3663, 4557, 4526, 2283, 3799 },
			  { 1, 3, 9, 27, 81, 243, 729, 2187 },
			  { 1, 1025, 4998, 4881, 25, 610, 4878, 1953, 625 },
			  { 1, 1611, 3767, 5001, 1781, 2472, 4, 1441, 59, 4995 }
			}, {
				{ 1 },
				{ 1 },
				{ 1, 4129 },
			  { 1, 544, 5733 },
			  { 1, 2425, 6516, 247 },
			  { 1, 4828, 3281, 9594, 7436 },
			  { 1, 3720, 8726, 8019, 9820, 4850 },
			  { 1, 4961, 4308, 7043, 5886, 20, 9157 },
			  { 1, 5002, 2504, 6251, 5634, 1556, 7673, 3501 },
			  { 1, 5, 25, 125, 625, 3125, 5618, 8076, 352 },
			  { 1, 4604, 1990, 5555, 7335, 6722, 6444, 7428, 4593, 1381 }
			}
	};

	/**
	 * Precalculated generating vectors for the method of embedded lattice rules
	 */
  static int[][][] zz_copy = {
			{
			  { 1 },
			  { 1 },
			  { 1, 544 },
			  { 1, 233, 436 },
			  { 1, 135, 71, 195 },
			  { 1, 18, 10, 23, 100 },
			  { 1, 27, 18, 12, 8, 58 },
			  { 1, 13, 5, 24, 25, 38, 2 },
			  { 1, 2, 4, 8, 16, 13, 7, 14 },
			  { 1, 2, 4, 8, 5, 10, 9, 7, 3 },
			  { 1, 2, 4, 3, 1, 2, 4, 3, 1, 2 }
			}, {
			  { 1 },
			  { 1 },
				  { 1, 920 },
				  { 1, 136, 1010 },
				  { 1, 73, 377, 285 },
				  { 1, 135, 71, 195, 33 },
				  { 1, 18, 10, 23, 100, 73 },
				  { 1, 27, 18, 12, 8, 58, 65 },
				  { 1, 19, 33, 12, 23, 27, 21, 30 },
				  { 1, 2, 4, 8, 16, 13, 7, 14, 9 },
				  { 1, 2, 4, 8, 5, 10, 9, 7, 3, 6 }
			}
	};
	
	/**
	 * Tests the implementation of all implemented integration rules.
	 * 
	 * @param output text area for presenting the results 
	 */
	public static void testImplementation(JTextArea output) {
		String outStr;
		PrintfFormat fd = new PrintfFormat("%12.6f");
		PrintfFormat fe = new PrintfFormat("%14.5e");
		PrintfFormat fi = new PrintfFormat("%8i");
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
		int s = 6;
		int m;
		int[] z;
		int i, j;
		int[] m_test = { 79, 157, 313, 619 /*, 1249, 2503, 5003*/ };
		EmbeddedCopyRule rule1 = new EmbeddedCopyRule();
		rule1.setUseErrorEstimation(true);
		GoodLatticePointsRule rule2 = new GoodLatticePointsRule();
		rule2.setUseErrorEstimation(true);
		Integrand myF = new FAlpha(s,2,FAlpha.ALPHA_2);
		output.append("Test of implementation\n\n");
		output.append("  ELR   - Embedded Lattice Rule\n");
		output.append("  GLPwE - Good Lattice Points Rule (with error estimation)\n");
		output.append("  GLP   - Good Lattice Points Rule (without error estimation)\n");
		output.append("  MC    - Monte Carlo\n\n");
		output.append("Method      m    2^6m     Qf      Error: true   estimated Runtime [s]   z\n");
		output.repaint();
		for (i = 0; i < m_test.length; i++)
		{
			long stopTime0 = System.currentTimeMillis();
			m = m_test[i];
			opt.setNCopy(2);
			z = opt.findOptimalZ(s,m);
			double q1 = rule1.evaluate(s,m,z,myF);
			output.append("ELR  "+fi.sprintf(m)+fi.sprintf(rule1.getNumberOfIntegrandEvaluations()));
			output.append(fd.sprintf(q1)+fd.sprintf(Math.abs(q1-1.0))+fd.sprintf(rule1.getEstimatedError()));
			long stopTime1 = System.currentTimeMillis();
			output.append(fd.sprintf((stopTime1-stopTime0)/1000.0)+" (");
			for (j=0;j < s; j++) {
				output.append(" "+z[j]);
			}
			output.append(" )\n");
			output.repaint();
			if (i < 5) {
				m = rule1.getNumberOfIntegrandEvaluations()/5;
				opt.setNCopy(1);
				z = opt.findOptimalZ(s,m);
				rule2.setUseErrorEstimation(true);
				q1 = rule2.evaluate(s,m,z,myF);
				output.append("GLPwE"+fi.sprintf(m)+fi.sprintf(rule2.getNumberOfIntegrandEvaluations()));
				output.append(fd.sprintf(q1)+fd.sprintf(Math.abs(q1-1.0))+fd.sprintf(rule2.getEstimatedError()));
				long stopTime2 = System.currentTimeMillis();
				output.append(fd.sprintf((stopTime2-stopTime1)/1000.0)+" (");
				for (j=0;j < s; j++) {
					output.append(" "+z[j]);
				}
				output.append(" )\n");
				output.repaint();
				//m = rule1.getNumberOfIntegrandEvaluations();
				opt.setNCopy(1);
				z = opt.findOptimalZ(s,m);
				rule2.setUseErrorEstimation(false);
				q1 = rule2.evaluate(s,m,z,myF);
				output.append("GLP  "+fi.sprintf(m)+fi.sprintf(rule2.getNumberOfIntegrandEvaluations()));
				output.append(fd.sprintf(q1)+fd.sprintf(Math.abs(q1-1.0))+fd.sprintf(rule2.getEstimatedError()));
				long stopTime3 = System.currentTimeMillis();
				output.append(fd.sprintf((stopTime3-stopTime2)/1000.0)+" (");
				for (j=0;j < s; j++) {
					output.append(" "+z[j]);
				}
				output.append(" )\n");
				output.repaint();
			}
			long stopTime4 = System.currentTimeMillis();
			MonteCarloIntegrator ruleMC = new MonteCarloIntegrator();
			m = rule1.getNumberOfIntegrandEvaluations();
			ruleMC.setEvaluationCountLimit(m);
			IntegrationResult q3 = ruleMC.integrate(myF);
			output.append("MC   "+fi.sprintf(m)+fi.sprintf(q3.getEvaluationCount()));
			output.append(fd.sprintf(q3.getValue())+fd.sprintf(Math.abs(q3.getValue()-1.0))+fd.sprintf(q3.getEstimatedError()));
			long stopTime5 = System.currentTimeMillis();
			output.append(fd.sprintf((stopTime5-stopTime4)/1000.0));
			output.append("\n\n");
			output.repaint();
		}
	}
	
	/**
	 * Tests the influence of dimension and order on the integration error.
	 * 
	 * @param output text area for presenting the results 
	 */
	public static void testDimension(JTextArea output) {
		PrintfFormat fe = new PrintfFormat("%14.5e");
		PrintfFormat fi = new PrintfFormat("%10i");
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
		opt.setNCopy(2);
		int m;
		int[] m_list = { 11, 19, 41, 79, 157, 313, 619, 1249, 2503, 5003 };
		int[] z;
		
		EmbeddedCopyRule rule = new EmbeddedCopyRule();
		rule.setUseErrorEstimation(true);
		output.setText("");
		output.append("Test of interdependencies of integrand dimension, lattice rule oder and integration error\n\n");
		output.append("         s         m    Error         Estimated   Err < Est\n");
		output.append("      ----  --------  ------------  ------------  ---------\n");
 		for (int s = 2; s <= 12; s++) {
  		Integrand myF = new FAlpha(s,2,FAlpha.ALPHA_2);
  		for (int j = 0; j < m_list.length; j++) {
				m = m_list[j];
				z = opt.findOptimalZ(s,m);
				double result = rule.evaluate(s, m, z, myF);
				double error = Math.abs(result-1.0);
				double estimate = rule.getEstimatedError();
				output.append(fi.sprintf(s)+fi.sprintf(rule.getNumberOfIntegrandEvaluations())+
						          fe.sprintf(error)+fe.sprintf(estimate)+"  "+(estimate > error)+"\n");
			}
  		output.append("\n");
		}
	}
	
	/**
	 * Test of choosing the generating vector z
	 * 
	 * @param output text area for presenting the results 
	 */
	
	public static void testChoice(JTextArea output) {
		PrintfFormat fe = new PrintfFormat("%14.5e");
		PrintfFormat fd = new PrintfFormat("%10.7f");
		PrintfFormat fi = new PrintfFormat("%3i");
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
		int s = 2;
		int m = 53;
		int[] z = new int[2];
		int[] z_opt;
		
		GoodLatticePointsRule rule1 = new GoodLatticePointsRule(new Trigonometric1Periodizer());
		Integrand myF1 = new SimpleTestIntegrands.F2DTest(s);
		GoodLatticePointsRule rule2 = new GoodLatticePointsRule();
		Integrand myF2 = new FAlpha(s,1,FAlpha.ALPHA_2);
		z_opt = opt.findOptimalZ(s,m);
		double result_opt = rule1.evaluate(s, m, z_opt, myF1);

		output.setText("");
		output.append("Test of choosing the generating vector\n\n");
		output.append(" z(opt)    = ("+fi.sprintf(z_opt[0])+","+fi.sprintf(z_opt[1])+" )\n");
		output.append(" If2d(opt) = "+fd.sprintf(result_opt)+"\n\n");
		output.append("  j    z          If_alpha    If2d        Error         Difference\n");
		output.append(" --  ---------  ----------  ----------  ------------  ------------\n");


		for (int j = 1; j < m; j++) {
			z[0] = 1;
			z[1] = j;
			double result1 = rule1.evaluate(s, m, z, myF1);
			double result2 = rule2.evaluate(s, m, z, myF2);
			double error = Math.abs(result1-1.0);
			double difference = result1 - result_opt;
			output.append(fi.sprintf(j)+"  ( 1,"+fi.sprintf(z[1])+" )  "+
					          fd.sprintf(result2)+"  "+fd.sprintf(result1)+fe.sprintf(error)+fe.sprintf(difference)+"\n");
		}
	}
	
	/**
	 * Test of the search for optimal lattice rules.
	 * 
	 * @param output text area for presenting the results 
	 */
	public static void testAlpha(JTextArea output) {
		PrintfFormat fi = new PrintfFormat("%7i");
		int m = 2503;
		int z[];
		Optimizer opt = new Optimizer();
		output.setText("");
		output.append("Search for optimal generating vectors in Korobov form\n\n");
		output.append("Rule order N    : 2503\n");
		output.append("Dimension s     : { 5, 10, 15, 20, 25, 50 }\n");
		output.append("Parameter Alpha : { 2, 4, 6 }\n");
		output.append("\n      s  alpha  z\n");
		output.append("   ----  -----  ---------------------- ...\n");
		int s = 5;
		while (s <= 50) {
			for (int alpha = FAlpha.ALPHA_2; alpha <= FAlpha.ALPHA_6; alpha += 2) {
				opt.setAlpha(alpha);
				z = opt.findOptimalZ(s,m);
				output.append(fi.sprintf(s)+fi.sprintf(alpha)+"  (");
				for (int j=0;j < s; j++) {
					output.append(" "+z[j]);
				}
				output.append(" )\n");
			}
			output.repaint();
			s += (s >= 25) ? 25 : 5;
		}
		
	}
	
	/**
	 * Tests the influence of periodizing the integrand.
	 * 
	 * @param output text area for presenting the results
	 * @param function selection of the integrand
	 *                 1: f(x,y) = y*exp(x*y)/(e-2)
	 *                 2: f(x,y) = 1 
	 */
	public static void testPeriodizing(JTextArea output, int function) {
		PrintfFormat fe = new PrintfFormat("%12.4e");
		PrintfFormat fi = new PrintfFormat("%7i");
		int s = 2;
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
		GoodLatticePointsRule rule1 = new GoodLatticePointsRule();
		GoodLatticePointsRule rule2 = new GoodLatticePointsRule(new Polynom1Periodizer());
		GoodLatticePointsRule rule3 = new GoodLatticePointsRule(new Polynom2Periodizer());
		GoodLatticePointsRule rule4 = new GoodLatticePointsRule(new Trigonometric1Periodizer());
		EmbeddedCopyRule rule5 = new EmbeddedCopyRule();
		EmbeddedCopyRule rule6 = new EmbeddedCopyRule(new Polynom1Periodizer());
		EmbeddedCopyRule rule7 = new EmbeddedCopyRule(new Polynom2Periodizer());
		EmbeddedCopyRule rule8 = new EmbeddedCopyRule(new Trigonometric1Periodizer());

		int[] m_test = { 79, 157, 313, 619, 1249, 2503, 5003, 10007 };
		int m;
		int[] z;
		double q1, q2, q3, q4, q5, q6, q7, q8;
   
		Integrand myIntegrand;
		if (function < 2) {
			myIntegrand = new SimpleTestIntegrands.F2DTest(s);
		} else {
			myIntegrand = new SimpleTestIntegrands.FOne(s);
		}
		output.setText("");
		output.append("Test of Periodization\n\n");
		output.append("  None     : phi(t)=t\n");
		output.append("  Period1  : phi(t)=(3-2*t)*t^2\n");
		output.append("  Period2  : phi(t)=t^3*(10-15*t+6*t^2)\n");
		output.append("  Trigonom1: phi(t)=t-sin(2*pi*t)/(2*pi)\n\n");
		output.append("  Integrand: "+myIntegrand.toString()+"\n");
		output.append("  Dimension: 2\n\n");
		output.append("                 Good Lattice Point Rule                         Embedded Lattice Rule\n");
		output.append("      m    4*m   None        Period1     Period2     Trigonom1   None        Period1     Period2     Trigonom1\n");
		for (int i = 0; i < m_test.length-2; i++) {
			m = m_test[i];
			opt.setNCopy(1);
			z = opt.findOptimalZ(s,m);
			q1 = Math.abs(rule1.evaluate(s,m,z,myIntegrand)-1.0);
			q2 = Math.abs(rule2.evaluate(s,m,z,myIntegrand)-1.0);
			q3 = Math.abs(rule3.evaluate(s,m,z,myIntegrand)-1.0);
			q4 = Math.abs(rule4.evaluate(s,m,z,myIntegrand)-1.0);
			opt.setNCopy(2);
			z = opt.findOptimalZ(s,m);
			q5 = Math.abs(rule5.evaluate(s,m,z,myIntegrand)-1.0);
			q6 = Math.abs(rule6.evaluate(s,m,z,myIntegrand)-1.0);
			q7 = Math.abs(rule7.evaluate(s,m,z,myIntegrand)-1.0);
			q8 = Math.abs(rule8.evaluate(s,m,z,myIntegrand)-1.0);
			output.append(fi.sprintf(m)+fi.sprintf(rule5.getNumberOfIntegrandEvaluations())+
					          fe.sprintf(q1)+fe.sprintf(q2)+fe.sprintf(q3)+fe.sprintf(q4)+
					          fe.sprintf(q5)+fe.sprintf(q6)+fe.sprintf(q7)+fe.sprintf(q8)+"\n");
			output.repaint();
		}
	}
	
	/**
	 * Tests the integration of integrands belonging to the Genz intgrand families.
	 * 
	 * @param output text area for presenting the results
	 * @param countFactor 
	 * @param periodize flag whether the integrand should be periodized of not
	 * @param integrandClass class of the integrand
	 * @param difficulty factor describing the desired complexity of the integrand
	 */
	public static void testGenz(JTextArea output, int countFactor, boolean periodize, int integrandClass, double difficulty) {
		PrintfFormat fd = new PrintfFormat("%9.2f");
		PrintfFormat fe = new PrintfFormat("%14.5e");
		PrintfFormat fi = new PrintfFormat("%8i");
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
	  GenzTestIntegrands gti = new GenzTestIntegrands();
	  int s;
	  Integrand f;
		int m_glp, m_copy, m_mc;
		int[] z_glp;
		int[] z_copy;
		int i, j;
		int[] m_list = { 80021, 40009, 20011, 10007, 5003, 2503, 1249, 619, 313, 157, 79, 41, 19, 11, 5 };
		double q1, d1, q2, d2, d3, sd1, sd2, sd3, e2, e3, se2, se3;
		IntegrationResult q3;

		int localIntegrandClass = Math.min(Math.max(0,integrandClass),gti.getNumberOfIntegrandClasses());
    f = gti.getIntegrand(localIntegrandClass,1,1.0);
		GoodLatticePointsRule rule1 = new GoodLatticePointsRule((periodize) ? new Trigonometric1Periodizer() : null);
		//rule1.setUseErrorEstimation(true);
		EmbeddedCopyRule rule2 = new EmbeddedCopyRule((periodize) ? new Trigonometric1Periodizer() : null);
		rule2.setUseErrorEstimation(true);
		MonteCarloIntegrator ruleMC = new MonteCarloIntegrator();
		int idx = Math.min(Math.max(0,countFactor),6);
		output.setText("");
		output.append("Genz-Test\n\n");
		output.append("  GLP   - Good Lattice Points Rule (without error estimation)\n");
		output.append("  ELR   - Embedded Lattice Rule\n");
		output.append("  MC    - Monte Carlo\n");
		output.append("\nType of integrand function: "+f.toString());
		output.append("\nNo. of integration points : about "+(Math.floor(Math.pow(2,idx)*5000)));
		output.append("\nPeriodizing integrand     : "+periodize);
		output.append("\nDifficulty of integrand   : "+difficulty+"\n");
		output.append("\n       s       m   GLP Error         m   ELR Error    Est.ok%   MC Error     Est.ok%");
		output.append("\n  ------  ------  ------------  ------  ------------  -------  ------------  -------\n");
	  for (s = 2+idx/3; s <= 10; s++) {
	  	if (idx > 0) {
				m_glp = 10007;
				z_glp = zz_glp[1][s];
	  	} else {
				m_glp = 5003;
				z_glp = zz_glp[0][s];
	  	}
			m_copy = m_list[4+s-idx];
			m_mc = (int) (Math.pow(2,s)*m_copy);
	  	if (idx > 1) {
				opt.setNCopy(2);
				z_copy = opt.findOptimalZ(s,m_copy);
	  	} else {
	  		z_copy = zz_copy[idx][s];
	  	}
			ruleMC.setEvaluationCountLimit(m_mc);
			sd1 = 0;
			sd2 = 0;
			sd3 = 0;
			se2 = 0;
			se3 = 0;
			for (j = 0; j < 20; j++) {
		    f = gti.getIntegrand(localIntegrandClass,s,difficulty);
				q1 = rule1.evaluate(s,m_glp,z_glp,f);
				d1 = Math.abs(1.0-q1);
				sd1 += d1;
				q2 = rule2.evaluate(s,m_copy,z_copy,f);
				d2 = Math.abs(1.0-q2);
				sd2 += d2;
				e2 = rule2.getEstimatedError();
				if (d2 < e2) se2++;
				q3 = ruleMC.integrate(f);
				d3 = Math.abs(1.0-q3.getValue());
				sd3 += d3;
				e3 = q3.getEstimatedError();
				if (d3 < e3) se3++;
			}
			sd1 /= 20;
			sd2 /= 20;
			sd3 /= 20;
			se2 *= 5;
			se3 *= 5;
			output.append(fi.sprintf(s)+fi.sprintf(m_glp)+fe.sprintf(sd1)+
					          fi.sprintf(rule2.getNumberOfIntegrandEvaluations())+
										fe.sprintf(sd2)+fd.sprintf(se2)+fe.sprintf(sd3)+fd.sprintf(se3)+"\n");
			output.repaint();
	  }
	}
	
	/**
	 * Test of Fibonacci lattice rules in 2 dimensions.
	 * 
	 * @param output text area for presenting the results 
	 */
	public static void testFibonacci(JTextArea output) {
		PrintfFormat fd = new PrintfFormat("%12.6f");
		PrintfFormat fe = new PrintfFormat("%12.4e");
		PrintfFormat fi = new PrintfFormat("%4i");
		int s = 2;
		int m;
		int[] z_fib = new int[2];
		int[] z_opt;
		double q1, q2, e1, e2;
		
		Integrand myF = new FAlpha(s,2,FAlpha.ALPHA_2);
		Optimizer opt = new Optimizer(FAlpha.ALPHA_2);
		GoodLatticePointsRule rule = new GoodLatticePointsRule();
		
		output.setText("");
		output.append("Fibonacci lattice rule ./. lattice rule in Korobov form\n\n");
		output.append("  GLP   - Good Lattice Points Rule (without error estimation)\n");
		output.append("  ELR   - Embedded Lattice Rule\n");
		output.append("  MC    - Monte Carlo\n");
		output.append("\n   k Fib(k)    z(Fib(k))  Error(Fib)    z(GLP)     Error(GLP)  Error diff." );
		output.append("\n --- ------  -----------  ----------  -----------  ----------  -----------\n");
		for (int k = 3; k < 20; k++) {
			z_fib[0] = 1;
			z_fib[1] = fibonacci(k-1);
			m = fibonacci(k);
			q1 = rule.evaluate(s,m,z_fib,myF);
			e1 = Math.abs(1.0-q1);
			z_opt = opt.findOptimalZ(s,m);
			q2 = rule.evaluate(s,m,z_opt,myF);
			e2 = Math.abs(1.0-q2);
			output.append(fi.sprintf(k)+"   "+fi.sprintf(m)+
					          "  ( 1, "+fi.sprintf(z_fib[1])+" )"+fe.sprintf(e1)+
					          "  ( 1, "+fi.sprintf(z_opt[1])+" )"+fe.sprintf(e2)+
										" "+fe.sprintf(e1-e2)+"\n");
		}
	}
	
	/**
	 * Calculates the k-th Fibonacci number.
	 * 
	 * @param k index of the requested Fibonacci number
	 * @return the requested Fibonacci number 
	 */
	public static int fibonacci(int k) {
	  int a, b, c, kk;
	  if ( k < 0 ) {
	    return -Integer.MAX_VALUE;
	  } else if ( k == 0 ) {
	    return 0;
	  } else if ( k == 1 ) {
	    return 1;
	  }
	  c = 0;
	  b = 0;
	  a = 1;
	  for (kk = 2; kk <= k; kk++) {
	    c = b;
	    b = a;
	    a = c + b;
	  }
	  return a;
	}
}
