/*
 * Created on 04.01.2005
 */
package q1471341.mp1074.test;

import java.util.Random;

import q1471341.mp1074.integration.Integrand;

/**
 * Generates test functions for benchmarking multidimensional integration
 * routines.
 * 
 * The test function classes are from: <br>
 * Alan Genz, Testing Multidimensional Integration Routines, Tools, Methods and
 * Languages for Scientific and Engineerings Computation, eds. B. Ford et. al,
 * 1984, pp. 81-94.
 * <p>
 * Each function class will produce functions of a specific type (gaussian,
 * oscillatory, etc.) with random parameters. These random parameters that are
 * generated using a <code>Random</code> object. This may be set, for example
 * with a seeded generator, so the randomization is reproducible.
 * <p>
 * All functions are normalized, that is the integral over the unit cube is 1.
 * 
 * @author Ulrich Telle
 */
public class GenzTestIntegrands {
	static public  final int INTEGRAND_CLASS_GENZ_OSCILLATORY   = 0;
	static public  final int INTEGRAND_CLASS_GENZ_PRODUCT_PEAK  = 1;
	static public  final int INTEGRAND_CLASS_GENZ_CORNER_PEAK   = 2;
	static public  final int INTEGRAND_CLASS_GENZ_GAUSSIAN      = 3;
	static public  final int INTEGRAND_CLASS_GENZ_CONTINUOUS    = 4;
	static public  final int INTEGRAND_CLASS_GENZ_DISCONTINUOUS = 5;
	static private final int NUMBER_OF_INTEGRAND_CLASSES        = 6;

	private Random random = new Random();

	/**
	 * Returns the number of integrand classes.
	 * 
	 * @return number of integrand classes
	 */
	public int getNumberOfIntegrandClasses() {
		return NUMBER_OF_INTEGRAND_CLASSES;
	}

	/**
	 * Returns a <code>String</code> description for a specific integrand class.
	 * <code>classNumber</code> must be between 0 and
	 * <code>getNumberOfIntegrandClasses() - 1</code>.
	 * 
	 * @param classNumber
	 *            number of the class
	 * @return description of the class
	 */
	public String getIntegrandClassName(int classNumber) {
		return getIntegrand(classNumber, 1, 1.0).toString();
	}

	/**
	 * Generate an integrand for a given integrand class. <code>classNumber</code>
	 * must be between 0 and <code>getNumberOfIntegrandClasses() - 1</code>.
	 * The difficulty specifies how difficult the integrand is to integrate
	 * compared to other integrands in this class. Obviously, difficulty depends
	 * on the integration algorithm used, so the difficulty levels are not at
	 * all comparable between integrand classes, and only heuristics within a
	 * integrand class.
	 * 
	 * @param classNumber
	 *            number of the class
	 * @param dimension
	 *            dimension of the integrand
	 * @param difficulty
	 *            difficulty of integrating the integrand
	 * @return test integrand
	 */
	public Integrand getIntegrand(int classNumber, int dimension,
			double difficulty) {
		if (classNumber == INTEGRAND_CLASS_GENZ_OSCILLATORY) {
			return new GenzOscillatory(dimension, difficulty);
		} else if (classNumber == INTEGRAND_CLASS_GENZ_PRODUCT_PEAK) {
			return new GenzProductPeak(dimension, difficulty);
		} else if (classNumber == INTEGRAND_CLASS_GENZ_CORNER_PEAK) {
			return new GenzCornerPeak(dimension, difficulty);
		} else if (classNumber == INTEGRAND_CLASS_GENZ_GAUSSIAN) {
			return new GenzGaussian(dimension, difficulty);
		} else if (classNumber == INTEGRAND_CLASS_GENZ_CONTINUOUS) {
			return new GenzContinuous(dimension, difficulty);
		} else if (classNumber == INTEGRAND_CLASS_GENZ_DISCONTINUOUS) {
			return new GenzDiscontinuous(dimension, difficulty);
		} else {
			throw new IllegalArgumentException("Integrand class number not valid");
		}
	}

	/**
	 * Sets the random generator for the random parameters of the functions.
	 * 
	 * @param random
	 *            random generator
	 */
	public void setRandom(Random random) {
		this.random = random;
	}

	public class GenzOscillatory extends Integrand {
		private int dimension;

		private double[] a;

		private double invResult;

		public GenzOscillatory(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			checkArgument(x);
			double s = 0.0;
			for (int i = 0; i < dimension; i++) {
				s += a[i] * x[i];
			}
			return Math.cos(2 * Math.PI * a[0] + s) * invResult;
		}

		private double result() {
			double product = 1.0;
			for (int i = 0; i < dimension; i++) {
				product *= Math.sin(0.5 * a[i]) * 2.0 / a[i];
			}
			return Math.cos(0.5 * (4 * Math.PI * a[0] + arraySum(a))) * product;
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Oscillatory";
		}
	}

	public class GenzProductPeak extends Integrand {
		private int dimension;

		private double[] a;

		private double[] u;

		private double invResult;

		public GenzProductPeak(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			u = randomU(dimension);
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			checkArgument(x);
			double p = 1.0;
			for (int i = 0; i < dimension; i++) {
				p *= 1.0 / (a[i] * a[i]) + (x[i] - u[i]) * (x[i] - u[i]);
			}
			return 1.0 / p * invResult;
		}

		private double result() {
			double p = 1.0;
			for (int i = 0; i < dimension; i++) {
				p *= a[i] * (Math.atan(a[i] * (1.0 - u[i])) + Math.atan(a[i] * u[i]));
			}
			return p;
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Product Peak";
		}
	}

	public class GenzCornerPeak extends Integrand {
		private int dimension;

		private double[] a;

		private double invResult;

		public GenzCornerPeak(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			checkArgument(x);
			double s = 0.0;
			for (int i = 0; i < dimension; i++) {
				s += a[i] * x[i];
			}
			return Math.pow((1.0 + s), -(dimension + 1)) * invResult;
		}

		private double result() {
			double p = 1.0;
			for (int i = 0; i < dimension; i++) {
				p *= a[i] * (i + 1);
			}
			return calcResult(1.0, 0) / p;
		}

		private double calcResult(double denominator, int i) {
			if (i == dimension) {
				return 1.0 / denominator;
			} else {
				return (calcResult(denominator, i + 1) - calcResult(denominator
						+ a[i], i + 1));
			}
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Corner Peak";
		}
	}

	public class GenzGaussian extends Integrand {
		private int dimension;

		private double[] a;

		private double[] u;

		private double invResult;

		public GenzGaussian(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			u = randomU(dimension);
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			checkArgument(x);
			double t, s = 0.0;
			for (int i = 0; i < dimension; i++) {
				t = a[i] * (x[i] - u[i]);
				s += t * t;
			}
			return Math.exp(-s) * invResult;
		}

		private double result() {
			double p = 1.0;
			final double sr2 = Math.sqrt(2.0);
			for (int i = 0; i < dimension; i++) {
				p *= (gaussian(sr2 * a[i] * (1 - u[i])) - gaussian(sr2 * a[i] * -u[i])) *
				     Math.sqrt(Math.PI) / a[i];
			}
			return p;
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Gaussian";
		}
	}

	public class GenzContinuous extends Integrand {
		private int dimension;

		private double[] a;

		private double[] u;

		private double invResult;

		public GenzContinuous(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			u = randomU(dimension);
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			checkArgument(x);
			double s = 0.0;
			for (int i = 0; i < dimension; i++) {
				s += a[i] * Math.abs(x[i] - u[i]);
			}
			return Math.exp(-s) * invResult;
		}

		private double result() {
			double p = 1.0;
			for (int i = 0; i < dimension; i++) {
				p *= (2.0 - Math.exp(-a[i] * u[i]) - Math.exp(-a[i] * (1.0 - u[i]))) / a[i];
			}
			return p;
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Continuous";
		}
	}

	public class GenzDiscontinuous extends Integrand {
		private int dimension;

		private double[] a;

		private double[] u;

		private double invResult;

		public GenzDiscontinuous(int dimension, double h) {
			this.dimension = dimension;
			a = randomA(dimension, h);
			u = randomU(2);
			// Prevent the function from being too singular
			u[0] = 0.05 + 0.95 * u[0];
			u[1] = 0.05 + 0.95 * u[1];
			invResult = 1.0 / result();
		}

		public double evaluate(double[] x) {
			if (x[0] > u[0] || (dimension >= 2 && x[1] > u[1])) {
				return 0.0;
			} else {
				double s = 0.0;
				for (int i = 0; i < dimension; i++) {
					s += a[i] * x[i];
				}
				return Math.exp(s) * invResult;
			}
		}

		private double result() {
			double p = 1.0;
			for (int i = 0; i < dimension; i++) {
				if (i < 2) {
					p *= (Math.exp(a[i] * u[i]) - 1) / a[i];
				} else {
					p *= (Math.exp(a[i]) - 1) / a[i];
				}
			}
			return p;
		}

		public int dimension() {
			return dimension;
		}

		public String toString() {
			return "Genz Discontinuous";
		}
	}

	// private methods
	
	private double[] randomU(int dim) {
		double[] u = new double[dim];
		for (int i = 0; i < u.length; i++) {
			u[i] = random.nextDouble();
		}
		return u;
	}

	/*
	 * The a values are distributed uniformily over the interval [0.05, 1[, and
	 * not [0,1[, as to small values in a component will cause a strongly
	 * singular function.
	 */
	private double[] randomA(int dim, double h) {
		double[] a = new double[dim];
		for (int i = 0; i < a.length; i++) {
			a[i] = 0.05 + 0.95 * random.nextDouble();
		}
		double normalizationFactor = h / arraySum(a);
		for (int i = 0; i < a.length; i++) {
			a[i] *= normalizationFactor;
		}
		return a;
	}

	/**
	 * Returns the sum over a <code>double</code> array.
	 * 
	 * @param x
	 *            array
	 * @return sum over x
	 */
	private static double arraySum(double[] x) {
		double s = 0.0;
		for (int i = 0; i < x.length; i++) {
			s += x[i];
		}
		return s;
	}

	/**
	 * Constants used in computing the cumulative gaussian distribution
	 */
	private static final double A0 = 0.398942270991, 
															A1 = 0.020133760596,
															A2 = 0.002946756074,

															B1 = 0.217134277847, 
															B2 = 0.018576112465, 
															B3 = 0.000643163695,

															C0 = 1.398247031184, 
															C1 = -0.360040248231, 
															C2 = 0.022719786588,

															D0 = 1.460954518699, 
															D1 = -0.305459640162, 
															D2 = 0.038611796258,
															D3 = -0.003787400686;

	/**
	 * Computes the cumulative distribution function for the Gaussian
	 * distribution.
	 * <p>
	 * The value is computed using an algorithm of Moro.
	 * 
	 * @param x
	 *            x
	 * @return p
	 */
	private static double gaussian(double x) {
		double y;
		double result;

		y = Math.abs(x);

		if (y <= 1.87) {
			double z = y * y;
			result = 0.5 + y * (A0 + (A1 + A2 * z) * z)
					/ (1.0 + (B1 + (B2 + B3 * z) * z) * z);
		} else if (y < 6.0) {
			result = 1.0 - Math.pow((C0 + (C1 + C2 * y) * y)
					/ (D0 + (D1 + (D2 + D3 * y) * y) * y), 16.0);
		} else {
			result = 1;
		}

		return (x >= 0.0) ? result : 1.0 - result;
	}
}