/*
 * Created on 10.01.2005
 */
package q1471341.mp1074.integration.lattice;

/**
 * This class optimizes generator vectors for lattice rules.
 * Only generator vectors of Korobov form are tested for optimality.
 * 
 * @author Ulrich Telle
 */
public class Optimizer {
	
	private LatticeRule latticeRule;

	int alpha;
	int nCopy;

	/**
	 * Constructs a default Optimizer.
	 * <code>nCopy=1</code> and <code>alpha=2</code>
	 */
	public Optimizer() {
		this(1, FAlpha.ALPHA_2);
	}
	
	/**
	 * Constructs an Optimizer with specified value for <code>alpha</code>.
	 * <code>nCopy=1</code>
	 */
	public Optimizer(int alpha){
		this(1, alpha);
	}
	
	/**
	 * Constructs an Optimizer with specified value for <code>nCopy> and <code>alpha</code>.
	 * An instance of the lattice rule 'Method of good lattice points' is created
	 * for internal use in the optimization.
	 */
	public Optimizer(int nCopy, int alpha){
		this.nCopy = nCopy;
		this.alpha = alpha;
		this.latticeRule = new GoodLatticePointsRule();
	}
	
	/**
	 * This method finds the appropriate generator vector <code>z</code>
	 * to minimize <code>P_alpha(Qf)</code.
	 * 
	 * For the method of good lattice points, a number of points <code>m</code>,
	 * and a single generator vector <code>z</code> is chosen.  The integrand is
	 * assumed to be periodic of period 1 in each argument, and is evaluated at
	 * each of the points x_i(1:s) = i * z(1:s) / m, for i = 0 to m-1.
	 * The integral is then approximated by the average of these values.
	 * 
	 * Assuming that s and m are known, and that the integrand is not
	 * known beforehand, the accuracy of the method depends entirely
	 * on the choice of z.  One method of choosing z is to search for
	 * the z among all candidates which minimizes a particular quantity
	 * P_alpha(Qf).
	 * 
	 * Here only the vectors z of the form (1, L, L^2, ..., L^(S-1)),
	 * for L = 1 to M/2 are checked.
	 * 
	 * @param s the spatial dimension.
	 * @param m the number of points to be used.
	 * @return z the optimal vector.
	 */
	public int[] findOptimalZ(int s, int m) {
	  int i;
		int l;
		double q0;
		double q0min;
		int value;
		int[] z = new int[s];
		int[] zMin = new int[s];
		FAlpha fAlpha = new FAlpha(s, nCopy, alpha);
		fAlpha.setCalculateDifference(true);

		q0min = Double.MAX_VALUE;
		for (l = 1; l <= m / 2; l++) {
			value = 1;
		  for (i = 0; i < s; i++) {
				z[i] = value;
		    value = (value * l) % m;
		  }

		  // Use this z and the lattice integral method Q0 of order m,
		  // to approximate the integral of P_alpha.
		  q0 = latticeRule.evaluate(s, m, z, fAlpha);

		  // If this result is the smallest so far, save the corresponding z.
		  if ( q0 < q0min ) {
		  	q0min = q0;
//		  	System.out.println("Opt: "+m+" "+l+" "+q0);
		  	for (i = 0; i < s; i++) {
		  	  zMin[i] = z[i];
		  	}
		  }
		}
		//  Return the best Z.
	  return zMin;
	}
	
	/**
	 * @return Returns the alpha value.
	 */
	public int getAlpha() {
		return alpha;
	}

	/**
	 * @param alpha The alpha to set.
	 */
	public void setAlpha(int alpha) {
		this.alpha = alpha;
	}
	
	/**
	 * @return Returns the nCopy.
	 */
	public int getNCopy() {
		return nCopy;
	}
	
	/**
	 * @param copy The nCopy to set.
	 */
	public void setNCopy(int copy) {
		nCopy = copy;
	}
}