/*
 * Created on 03.01.2005
 */
package q1471341.mp1074.integration.lattice;

import q1471341.mp1074.integration.Integrand;

/**
 * The abstract root class of the lattice rule hierarchy.
 * 
 * The subclasses implement specific lattice rules for numerical integration.
 * 
 * @author Ulrich Telle
 */
public abstract class LatticeRule {

	protected int        nCopy;
	protected boolean    useErrorEstimation;
	protected double     estimatedError;
	protected int        numberOfIntegrandEvaluations;
	protected Periodizer periodizer; 

	/**
	 * Constructs an instance of a lattice rule.
	 * 
	 * Error estimation and periodizing of the integrand function
	 * is turned off by default. 
	 */
	public LatticeRule() {
		nCopy = 1;
		useErrorEstimation = false;
		estimatedError = 0.0;
		numberOfIntegrandEvaluations = 0;
		periodizer = null;
	}
	
	/**
	 * Performs evaluation of the lattice rule.
	 * 
	 * @param s dimension of the integrand domain
	 * @param m order of the lattice rule
	 * @param z lattice rule generator vector
	 * @param f integrand function
	 * @return result of integration
	 */
	public abstract double evaluate(int s, int m, int[] z, Integrand f);
	
	/**
	 * @param useErrorEstimation The error estimation flag to set.
	 */
	public void setUseErrorEstimation(boolean useErrorEstimation) {
		this.useErrorEstimation = useErrorEstimation;
	}

	/**
	 * @return Returns the flag whether error estimation should be performed.
	 */
	public boolean getUseErrorEstimation() {
		return useErrorEstimation;
	}

	/**
	 * @return Returns the estimated error of integral value.
	 */
	public double getEstimatedError() {
		return estimatedError;
	}
	
	/**
	 * @return Returns the number of integrand evaluations.
	 */
	public int getNumberOfIntegrandEvaluations() {
		return numberOfIntegrandEvaluations;
	}

	/**
	 * @return Returns the periodizer.
	 */
	public Periodizer getPeriodizer() {
		return periodizer;
	}
	
	/**
	 * @param periodizer The periodizer to set.
	 */
	public void setPeriodizer(Periodizer periodizer) {
		this.periodizer = periodizer;
	}
	
	/**
	 * @return Returns the nCopy.
	 */
	public int getNCopy() {
		return nCopy;
	}
}
