/*
 * Created on 03.01.2005
 */
package q1471341.mp1074.integration.lattice;

import q1471341.mp1074.integration.Integrand;

/**
 * Implements the lattice rule 'Method of good lattice points'.
 * 
 * @author Ulrich Telle
 */
public class GoodLatticePointsRule extends LatticeRule {

	/**
	 * Number of evaluations if error estimation by randomization is performed.
	 */
  private static final int Q_MAX = 5;
	
  /**
   * Constructs an instance of a lattice rule 'Method of good lattice points'.
   */
	public GoodLatticePointsRule() {
		super();
	}

  /**
   * Constructs an instance of a lattice rule 'Method of good lattice points'.
   * 
   * @param periodizer Periodizer to use for integrand function transformation
   */
	public GoodLatticePointsRule(Periodizer periodizer) {
		super();
		this.periodizer = periodizer;
	}
	
	/**
	 * Performs evaluation of the lattice rule.
	 * 
	 * Because this is a standard lattice rule, it is really only suited
	 * for functions which are periodic, of period 1, in all dimensions.
	 * 
	 * For a suitable integrand <code>f</code>, and a given value of
	 * <code>m</code> (the number of lattice points), the performance of
	 * the routine is affected by the choice of the generator vector <code>z</code>.
	 * 
	 * @param s the dimension of the integrand domain
	 * @param m the order of the lattice rule
	 * @param z the lattice rule generator vector.  Typically, the elements
	 * of <code>z</code> satisfy <code>1 <= z_i < m</code>, and are relatively
	 * prime to <code>m</code>. This is easy to guarantee if <code>m</code> is
	 * itself a prime number.
	 * @param f the user-supplied integrand function
	 * @return the estimated integral of f over the unit hypercube.
	 * 
	 * Note: Error estimation is performed by randomization if requested.
	 */
	public double evaluate(int s, int m, int[] z, Integrand f) {
	  int j;
	  double quad;
	  double[] x = new double[s];

	  if (useErrorEstimation) {
			numberOfIntegrandEvaluations = 0;
	  	quad = estimateError(s, m, z, f);
	  } else {
		  quad = 0.0;
		  for (j = 0; j < m; j++) {
		  	for ( int k = 0; k < s; k++) {
		  		x[k] = ((double)(j*z[k]) / (double) (m)) % 1.0;
		  	}
				if (periodizer != null) {
					periodizer.periodize(x);
					quad += f.evaluate(x)*periodizer.getDerivative();
				} else {
					quad += f.evaluate(x);
				}
	    }
		  quad = quad / ((double) m );
			numberOfIntegrandEvaluations = m;
	  	estimatedError = 0.0;
	  }
	  return quad;
	}
	
	/**
	 * Estimates the integration error by randomization
	 * 
	 * @param s spatial dimension
	 * @param m order of lattice rule
	 * @param z generating vector
	 * @param f the integrand function
	 * @return
	 */
	private double estimateError(int s, int m, int[] z, Integrand f) {
    int q;
    int j, k;
    double[] qc = new double[Q_MAX];
    double qcMean, qcSum, difference;
	  double[] x = new double[s];
	  double[] c = new double[s];
  	for (k = 0; k < s; k++) {
  		c[k] = 0;
  	}
    
	  qcMean = 0.0;
	  for (q = 0; q < Q_MAX; q++) {
	  	for (k = 0; k < s; k++) {
	  		c[k] = Math.random();
	  	}
	    qc[q] = 0.0;
	    for (j = 0; j < m; j++) {
	  	  for (k = 0; k < s; k++) {
	  		  x[k] = ((double)(j*z[k]) / (double) (m) + c[k]) % 1.0;
	  	  }
				if (periodizer != null) {
					periodizer.periodize(x);
					qc[q] += f.evaluate(x)*periodizer.getDerivative();
				} else {
					qc[q] += f.evaluate(x);
				}
	    }
	    qc[q] /= ((double) m );
	    qcMean += qc[q];
			numberOfIntegrandEvaluations += m;
	  }
	  qcMean /= Q_MAX;
	  qcSum = 0.0;
	  for (q = 0; q < Q_MAX; q++) {
	  	difference = qc[q] - qcMean; 
	  	qcSum += (difference*difference);
	  }
	  estimatedError = Math.sqrt(qcSum / (Q_MAX*(Q_MAX-1))); 
	  return qcMean;
	}
}
