/*
 * Created on 15.01.2005
 */
package q1471341.mp1074.integration.lattice;

import q1471341.mp1074.integration.Integrand;

/**
 * This class implements the integrand function usually used in the
 * optimization of generator vectors for lattice rules.
 * 
 * Optimization is possible for three predefined values of alpha, namely
 * alpha in { 2, 4, 6 }. The value of nCopy allows to optimize not only 
 * generator vectors for good lattice points lattice rules, but also for
 * embedded copy rules.
 * 
 * @author Ulrich Telle
 */
public class FAlpha extends Integrand {

	public static final int ALPHA_2 = 2;
	public static final int ALPHA_4 = 4;
	public static final int ALPHA_6 = 6;

	private int s;
	private int nCopy;
	private int alpha;
	private boolean calculateDifference;

	/**
	 * Constructs an instance of the integrand fucntion <code>F_Alpha</code>
	 * 
	 * @param dim  the spatial dimension.
	 */
	public FAlpha(int dim) {
		s = dim;
		this.nCopy = 1;
		this.alpha = ALPHA_2;
		this.calculateDifference = false;
	}

	/**
	 * Constructor of class <code>F_Alpha</code>
	 * 
	 * @param dim  the spatial dimension.
	 * @param nCopy copy value of embedded lattice rule if applicable
	 * @param alpha requested goodness of approximation
	 */
	public FAlpha(int dim, int nCopy, int alpha) {
		s = dim;
		this.nCopy = nCopy;
		this.alpha = alpha;
	}

	/**
	 * @return Returns the dimension of the integrand domain
	 */
	public int dimension() {
		return s;
	}

	/**
	 * Evaluates the integrand function at a given point x.
	 * 
	 * @param x point at which the integrand is to be evaluated
	 * @return value of the integrand function.
	 */
	public double evaluate(double[] x) {
		checkArgument(x);
		int i;
		int n;
		double result = 1.0;
		double divisor = 1.0;

		switch (alpha) {
		case ALPHA_6:
			divisor = Math.pow(nCopy, 6);
			for (i = 0; i < s; i++) {
				result = result * (1.0 + (F6(x[i]) - 1.0) / divisor);
			}
			break;
		case ALPHA_4:
			divisor = Math.pow(nCopy, 4);
			for (i = 0; i < s; i++) {
				result = result * (1.0 + (F4(x[i]) - 1.0) / divisor);
			}
			break;
		case ALPHA_2:
		default:
			divisor = Math.pow(nCopy, 2);
			for (i = 0; i < s; i++) {
				result = result * (1.0 + (F2(x[i]) - 1.0) / divisor);
			}
			break;
		}
		if (calculateDifference) {
			result = result - 1.0;
		}
		return result;
	}

	/**
	 * @return Returns a description of the integrand.
	 */
	public String toString() {
		return "Q_0 f_alpha - 1";
	}
	
	/**
	 * F2 evaluates a function of a scalar used in defining P2(Q).
	 * 
	 * @param x   the value of the argument
	 * @return the value of F2(X)
	 */
	private static double F2(double x) {
		double result = 1.0 + 2.0 * Math.PI * Math.PI * (x * x - x + 1.0 / 6.0);
		return result;
	}

	/**
	 * F4 evaluates a function of a scalar used in defining P4(Q).
	 * 
	 * @param x   the value of the argument
	 * @return the value of F4(X)
	 */
	private static double F4(double x) {
		double result = 1.0 + Math.pow(Math.PI, 4.0) / 45.0
				* (1.0 - 30.0 * x * x * (1.0 - x) * (1.0 - x));
		return result;
	}

	/**
	 * F6 evaluates a function of a scalar used in defining P6(Q).
	 * 
	 * @param x   the value of the argument
	 * @return the value of F2(X)
	 */
	private static double F6(double x) {
		double result = 1.0 + 2.0 * Math.pow(Math.PI, 6.0) / 945.0
				* (1.0 + x * x * (-21.0 + x * x * (105.0 + x * (-126.0 + 42.0 * x))));
		return result;
	}
	
	/**
	 * @param calculateDifference The calculateDifference to set.
	 * 
	 * Note: This flag is only used while optimizing generator vectors for
	 * lattice rules.
	 */
	public void setCalculateDifference(boolean calculateDifference) {
		this.calculateDifference = calculateDifference;
	}
}