/*
 * Created on 10.01.2005
 */
package q1471341.mp1074.integration.lattice;

import q1471341.mp1074.integration.Integrand;

/**
 * Implements the lattice rule 'Embedded copy sequence of lattice rules'.
 * 
 * @author Ulrich Telle
 */
public class EmbeddedCopyRule extends LatticeRule {

	private final int DEFAULT_NCOPY = 2;
	
  /**
   * Constructs an instance of a lattice rule <code>EmbeddedCopyRule</code>.
   */
	public EmbeddedCopyRule() {
		super();
		nCopy = DEFAULT_NCOPY;
	}

  /**
   * Constructs an instance of a lattice rule <code>EmbeddedCopyRule</code>.
   * 
   * @param periodizer Periodizer to use for integrand function transformation
   */
	public EmbeddedCopyRule(Periodizer periodizer) {
		super();
		this.periodizer = periodizer;
	}

	/**
	 * Performs evaluation of the lattice rule.
	 * 
	 * Because this is a standard lattice rule, it is really only suited
	 * for functions which are periodic, of period 1, in all dimensions.
	 * 
	 * For a suitable integrand <code>f</code>, and a given value of
	 * <code>m</code> (the number of lattice points), the performance of
	 * the routine is affected by the choice of the generator vector <code>z</code>.
	 * 
	 * @param s the dimension of the integrand domain
	 * @param m the order of the lattice rule. Since the copy value <code>n</code>
	 * is set to 2 in this implementation, <code>m</code> has to be odd.
	 * Integration and error estimation might fail if this condition is not met.
	 * @param z the lattice rule generator vector.  Typically, the elements
	 * of <code>z</code> satisfy <code>1 <= z_i < m</code>, and are relatively
	 * prime to <code>m</code>. This is easy to guarantee if <code>m</code> is
	 * itself a prime number.
	 * 
	 * @param f the user-supplied integrand function
	 * @return the estimated integral of f over the unit hypercube.
	 */
	public double evaluate(int s, int m, int[] z, Integrand f) {
		double result = 0;
		int[] k = new int[s+1];
		double[] x = new double[s];
		int n = DEFAULT_NCOPY;
		double[] q = new double[s+1];
		double val;
		int i, j, l;
		int omega;
		double[] qq = new double[s+1];
		double e;
		
		numberOfIntegrandEvaluations = 0;
		q[0] = 0.0;
		for (i=1; i <= s; i++){
			k[i] = 0;
			q[i] = 0.0;
			qq[i] = 0.0;
		}
		k[1] = -1;
		l = 1;
		omega = 0;
		while (l <= s) {
			if (k[l] < n-1) {
				k[l]++;
				l = 1;
				val = 0.0;
				for (j=0; j < m; j++) {
					for (i=1; i <= s; i++) {
						x[i-1] = (((double) j * z[i-1])/((double) m) + ((double) k[i]/ (double) n)) % 1.0;
					}
					if (periodizer != null) {
						periodizer.periodize(x);
						val += f.evaluate(x)*periodizer.getDerivative();
					} else {
						val += f.evaluate(x);
					}
				}
				numberOfIntegrandEvaluations += m;
				for (i = omega; i <= s; i++) {
					q[i] += val;
				}
				for (i = 1; i <= s; i++) {
					if (k[i] == 0) {
						qq[i] += val;
					}
				}
				if (omega == 0) {
					omega = 1;
				}
			} else {
				k[l] = 0;
				l++;
				if (omega < l) {
					omega = l;
				}
			}
		}
		int npot = 1;
		for (i = 0; i <= s; i++) {
			q[i] = q[i] / (npot*m);
			npot *= n;
		}
		// npot now equals n^(s+1), but n^(s-1) is needed in the following loop
		npot /= (n*n);
		
		estimatedError = 0.0;
		if (useErrorEstimation) {
			double difference;
			for (i = 1; i <= s; i++) {
				qq[i] = qq[i] / (npot*m);
				difference = q[s]-qq[i];
				estimatedError += (difference*difference);
			}
			estimatedError = Math.sqrt(estimatedError/s);
		}
		return q[s];
	}
}
