/*
 * Created on 03.01.2005
 */
package q1471341.mp1074.integration;

import q1471341.mp1074.integration.lattice.GoodLatticePointsRule;
import q1471341.mp1074.integration.lattice.LatticeRule;
import q1471341.mp1074.integration.lattice.Optimizer;

/**
 * Performs a lattice rule integration.
 * 
 * @author Ulrich Telle
 */
public class LatticeIntegrator extends Integrator {
	
	private Optimizer   optimizer;
	private LatticeRule rule;
	
	/**
	 * Instantiates a lattice rule integrator
	 * 
	 * @param latticeRule thelattice rule to use
	 *                    if not specified, e.g. <code>null</code>,
	 *                    a good lattice points rule is used
	 * @param optimizer the optimizer instance used to find the optimal
	 *                    lattice generating vector  
	 *                    if not specified, e.g. <code>null</code>,
	 *                    the default optimizer is used
	 */
	public LatticeIntegrator(LatticeRule latticeRule, Optimizer optimizer) {
		if (rule != null) {
	    this.rule = latticeRule;
		} else {
			this.rule = new GoodLatticePointsRule();
		}
		if (optimizer != null) {
			this.optimizer = optimizer;
		} else {
			this.optimizer = new Optimizer();
		}
		this.optimizer.setNCopy(this.rule.getNCopy());
	}
	
	/**
	 * Performs numerical integration of the integrand until one of the
	 * termination conditions is met.
	 * 
	 * @param f integrand function
	 * @return result of integration
	 * @throws IntegrationException if an integration error occurs
	 */
	public IntegrationResult integrate(Integrand f) {
		int dimension = f.dimension();
		int m = 1;
		double value;
		long evaluationCount = 0;
		
		startTimer();
		do {
			if (m < 1000) {
				m += 100;
			} else {
				m += 1000;
			}
			int[] z = optimizer.findOptimalZ(dimension, m);
			value = rule.evaluate(dimension, m, z, f);
			evaluationCount += rule.getNumberOfIntegrandEvaluations();
		}	while (!shouldTerminate(value,evaluationCount));
		
		IntegrationResult result = new IntegrationResult(value,rule.getEstimatedError(),evaluationCount,getEvaluationTime(),getTerminationCondition());
		return result;
	}

	/**
	 * @return Returns the name of this integrator
	 */
	public String toString() {
		return "Lattice Rule Integrator";
	}
}
