/*
 * Created on 03.01.2005
 */
package q1471341.mp1074.integration;

/**
 * The abstract root class of the integrator hierarchy.
 * 
 * The subclasses implement specific algorithms for numerical integration.
 * The integration is always performed over the unit cube <i>[0,1]^s</i>,
 * where <i>s</i> is the dimension of the integrand.
 * <p>
 * Only the abstract method <code>integrate</code> needs to be implemented
 * by subclasses. In the <code>integrate</code> method, subclasses should
 * continually refine the result of the integration. Regularly the method
 * <code>shouldTerminate</code> should be called, which checks whether one
 * or more of the termination conditions is met.
 * Currently 4 termination conditions are implemented:
 * <ul>
 * <li>absolute tolerance - integration stops if two successive integration
 * results differ from each other no more than the specified absolute tolerance,</li>
 * <li>relative tolerance - integration stops if two successive integration
 * results differ from each other no more than the specified relative tolerance,</li>
 * <li>evaluation count limit - integration stops if the integrand was evaluated 
 * more often than the specified evaluation count limit,</li>
 * <li>evaluation time limit - integration stops if more time elapsed since the
 * start of the integration than the specified evaluation time limit.</li>
 * </ul>
 * 
 * The final result of the integration process is returned in an
 * <code>IntegrationResult</code> instance.
 * 
 * @author Ulrich Telle
 */
public abstract class Integrator {
	
	/**
	 * Termination condition constants
	 */
	public final int TC_ABSOLUTE_TOLERANCE = 1;
	public final int TC_RELATIVE_TOLERANCE = 2;
	public final int TC_EVALUATION_COUNT   = 3;
	public final int TC_EVALUATION_TIME    = 4;
	
	/**
	 * Default relative tolerance
	 */
	private final double DEFAULT_RELATIVE_TOLERANCE = 1e-6;

	private double absoluteTolerance;
	private double relativeTolerance;
	private long   evaluationCountLimit;
	private long   evaluationTimeLimit;
	private long   startTime;
	private double lastQuadratureValue;
	private long   timeOut;
	
	private long   evaluationTime;
	private int    terminationCondition;

	/**
	 * Performs numerical integration of the function until one of the
	 * termination conditions is met.
	 * 
	 * @param function integrand function
	 * @return result of integration
	 * @throws IntegrationException if an integration error occurs
	 */
	public abstract IntegrationResult integrate(Integrand function) throws IntegrationException;

	/**
	 * @return Returns the absoluteTolerance.
	 */
	public double getAbsoluteTolerance() {
		return absoluteTolerance;
	}

	/**
	 * @param absoluteTolerance The absoluteTolerance to set.
	 */
	public void setAbsoluteTolerance(double absoluteTolerance) {
		this.absoluteTolerance = absoluteTolerance;
	}
	
	/**
	 * @return Returns the evaluationCountLimit.
	 */
	public long getEvaluationCountLimit() {
		return evaluationCountLimit;
	}
	
	/**
	 * @param evaluationCountLimit The evaluationCountLimit to set.
	 */
	public void setEvaluationCountLimit(long evaluationCountLimit) {
		this.evaluationCountLimit = evaluationCountLimit;
	}
	
	/**
	 * @return Returns the relativeTolerance.
	 */
	public double getRelativeTolerance() {
		return relativeTolerance;
	}
	
	/**
	 * @param relativeTolerance The relativeTolerance to set.
	 */
	public void setRelativeTolerance(double relativeTolerance) {
		this.relativeTolerance = relativeTolerance;
	}
	
	/**
	 * @return Returns the terminationCondition.
	 */
	public int getTerminationCondition() {
		return terminationCondition;
	}

	/**
	 * @return Returns the evaluationTime.
	 */
	public long getEvaluationTime() {
		return evaluationTime;
	}

	/**
	 * Constructs an instance of an abstract integrator
	 * using default values for the termination conditions.
	 */
	protected Integrator() {
		absoluteTolerance = 0.0;
		relativeTolerance = DEFAULT_RELATIVE_TOLERANCE;
		evaluationCountLimit = 0;
		evaluationTimeLimit = 0;
	}
	
	/**
	 * Starts the timer at the beginning of the integration process.
	 */
	protected void startTimer() {
		startTime = System.currentTimeMillis();
		timeOut = startTime + evaluationTimeLimit;
	}

	/**
	 * Checks the termination conditions.
	 * 
	 * @param quadratureValue current result of the integration
	 * @param evaluationCount current number of integrand evaluations
	 * @return <code>true</code> if at least on termination condition is met,
	 * otherwise <code>false</code>.
	 */
	protected boolean shouldTerminate(double quadratureValue, long evaluationCount) {
		double tolerance;
		if (absoluteTolerance > 0.0) {
			tolerance = Math.abs(quadratureValue - lastQuadratureValue);
      if (tolerance < absoluteTolerance) {
				terminationCondition = TC_ABSOLUTE_TOLERANCE;
				return true;
      }
		}
		if (relativeTolerance > 0.0) {
			tolerance = Math.abs(quadratureValue - lastQuadratureValue) / Math.abs(quadratureValue);
      if (tolerance < absoluteTolerance) {
				terminationCondition = TC_RELATIVE_TOLERANCE;
				return true;
      }
		}
		if (evaluationCountLimit > 0) {
			if (evaluationCount >= evaluationCountLimit) {
				terminationCondition = TC_EVALUATION_COUNT;
				return true;
			}
		}
		if (evaluationTimeLimit > 0) {
			if (System.currentTimeMillis() >= timeOut) {
				evaluationTime = System.currentTimeMillis() - startTime;
				terminationCondition = TC_EVALUATION_TIME;
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Checks whether at least one termination conditions was specified.
	 * 
	 * @throws IntegrationException
	 */
	private void checkConditions() throws IntegrationException {
		if (absoluteTolerance <= 0.0 && relativeTolerance <= 0.0 &&
				evaluationCountLimit <= 0 && evaluationTimeLimit <= 0) {
			throw new IntegrationException("No termination condition specified.");
		}
	}
}
